/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.map.proj;

import gov.nasa.giss.graphics.GraphicUtils;
import gov.nasa.giss.img.ImageUtils;
import gov.nasa.giss.map.LonLatEdges;
import gov.nasa.giss.map.MapUtils;
import gov.nasa.giss.map.SymbolFactory;
import gov.nasa.giss.map.SymbolID;
import gov.nasa.giss.map.overlay.OutlineArea;
import gov.nasa.giss.map.overlay.OutlineOverlay;
import gov.nasa.giss.map.overlay.OutlineSegment;
import gov.nasa.giss.map.proj.ProjBooleanParameter;
import gov.nasa.giss.map.proj.ProjDoubleParameter;
import gov.nasa.giss.map.proj.ProjExtraParameter;
import gov.nasa.giss.map.proj.ProjIntegerParameter;
import gov.nasa.giss.map.proj.ProjListParameter;
import gov.nasa.giss.map.proj.ProjParameterEvent;
import gov.nasa.giss.map.proj.ProjParameterListener;
import gov.nasa.giss.map.proj.ProjectionUtils;
import gov.nasa.giss.map.proj.ui.ProjBooleanComponent;
import gov.nasa.giss.map.proj.ui.ProjDoubleComponent;
import gov.nasa.giss.map.proj.ui.ProjIntegerComponent;
import gov.nasa.giss.map.proj.ui.ProjListComponent;
import gov.nasa.giss.map.proj.ui.ProjParamComponent;
import gov.nasa.giss.math.PointLL;
import gov.nasa.giss.text.PrintfFormat;
import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.geom.Area;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.MemoryImageSource;
import java.lang.invoke.MethodHandles;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractProjection
implements ProjParameterListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    protected static final int ITER_MAX = 33;
    protected static final double SMALL_VALUE = 1.0E-5;
    protected static final double ALMOST_90 = 89.99999;
    protected static final double ALMOST_M90 = -89.99999;
    protected static final double ALMOST_180 = 179.99999;
    protected static final double TINY_VALUE = 1.0E-10;
    private static final PrintfFormat GRID_LABEL_FORMATTER = new PrintfFormat("%.5G");
    private int[] srcPixels_;
    private int srcWidth_ = 1;
    private int srcHeight_ = 1;
    private double srcCenterX_;
    private double srcCenterY_;
    private double srcCenterLon_;
    private double srcCenterLat_;
    private double srcWidthDeg_;
    private double srcHeightDeg_;
    private double srcPixelPerDegLon_;
    private double srcPixelPerDegLat_;
    private int outWidth_ = 2;
    private int outHeight_ = 2;
    protected int outCenterX_;
    protected int outCenterY_;
    private int lMargin_;
    private int rMargin_;
    private int tMargin_;
    private int bMargin_;
    private boolean needsInverseRefresh_ = true;
    private int[] invArray_;
    private double[] invArrayLon_;
    private double[] invArrayLat_;
    protected double lambdaC_;
    protected double lambdaCRad_;
    protected double phiC_;
    protected double phiCRad_;
    private double zoom_ = 1.0;
    private boolean zoomEnabled_;
    protected double rS_ = 1.0;
    protected double r2s2_ = 1.0;
    protected double invRS_ = 1.0;
    protected int dxMax_;
    protected int dyMax_;
    protected double xmRS_;
    protected double ymRS_;
    private double xMaxOverRS_ = 1.0;
    private double yMaxOverRS_ = 1.0;
    private final String name_;
    private final int properties_;
    private ArrayList<ProjExtraParameter> extraParams_;
    private static HashMap<ProjExtraParameter, ProjParamComponent> paramGuiHash_ = new HashMap(5);
    private Color background_ = Color.WHITE;
    private Color borderColor_ = Color.BLACK;
    private BasicStroke borderStroke_ = new BasicStroke(1.75f);
    private Color gridColor_ = Color.GRAY;
    private BasicStroke gridStroke_ = new BasicStroke(1.0f);
    private double lonGridSpacing_ = 30.0;
    private double latGridSpacing_ = 30.0;
    private boolean latGridOffset_;
    private boolean gridLabeled_;
    private int gridLabelStep_ = 1;
    private Font gridFont_ = new Font("SansSerif", 0, 10);
    private Color pathColor_ = Color.RED;
    private BasicStroke pathStroke_ = new BasicStroke(1.0f, 1, 1);
    private SymbolID ptSymbol_ = SymbolID.SALTIRE;
    private float ptSymbolSize_ = 3.0f;
    private boolean constructed_;

    private AbstractProjection() {
        throw new UnsupportedOperationException("Use a constructor that has arguments");
    }

    protected AbstractProjection(String name, int props, int width, int height, int xmargin, int ymargin, double widthFactor, double heightFactor) {
        this.name_ = name;
        this.properties_ = props;
        this.setCenter(0.0, 0.0);
        this.setMaxXYOverRS(widthFactor, heightFactor);
        this.setSizeAndMargins(width, height, xmargin, ymargin);
    }

    protected final void finishConstruction() {
        this.parameterChanged(null);
        this.constructed_ = true;
        this.autoscale();
    }

    public int getWidth() {
        return this.outWidth_;
    }

    public int getHeight() {
        return this.outHeight_;
    }

    public void setSizeAndMargins(int w, int h, int xm, int ym) {
        if (xm < 0) {
            throw new IllegalArgumentException("X margin cannot be negative.");
        }
        if (ym < 0) {
            throw new IllegalArgumentException("Y margin cannot be negative.");
        }
        this.invArray_ = null;
        this.outWidth_ = Math.max(w, 8);
        this.outHeight_ = Math.max(h, 4);
        this.lMargin_ = xm;
        this.tMargin_ = ym;
        if (this.outWidth_ < 2 * xm + 8) {
            throw new IllegalArgumentException("Width must be greater than 2*xm+8");
        }
        if (this.outHeight_ < 2 * ym + 4) {
            throw new IllegalArgumentException("Height must be greater than 2*ym+4");
        }
        this.rMargin_ = this.outWidth_ - xm;
        this.bMargin_ = this.outHeight_ - ym;
        this.outCenterX_ = (int)(0.5 * (double)w);
        this.outCenterY_ = (int)(0.5 * (double)h);
        this.autoscale();
    }

    public Insets getInsets() {
        return new Insets(this.tMargin_, this.lMargin_, this.tMargin_, this.lMargin_);
    }

    protected void setMaxXYOverRS(double sizeFactor) {
        this.setMaxXYOverRS(sizeFactor, sizeFactor);
    }

    protected void setMaxXYOverRS(double wf, double hf) {
        this.xMaxOverRS_ = wf;
        this.yMaxOverRS_ = hf;
    }

    protected final void autoscale() {
        if (!this.constructed_) {
            return;
        }
        this.prepareScaling();
        this.calculateScaling();
        this.finishScaling();
        this.needsInverseRefresh_ = true;
    }

    protected void prepareScaling() {
    }

    private void calculateScaling() {
        int uw = this.outWidth_ - 2 * this.lMargin_;
        int uh = this.outHeight_ - 2 * this.tMargin_;
        double r = 0.5 * Math.min((double)uw / this.xMaxOverRS_, (double)uh / this.yMaxOverRS_);
        this.rS_ = r * this.zoom_;
        this.invRS_ = 1.0 / this.rS_;
        this.r2s2_ = this.rS_ * this.rS_;
        this.xmRS_ = this.xMaxOverRS_ * this.rS_;
        this.ymRS_ = this.yMaxOverRS_ * this.rS_;
        if (this.hasProperty(64) || this.hasProperty(16) || this.hasProperty(128) || this.hasProperty(2048)) {
            this.dxMax_ = (int)(0.5 * (double)uw + 0.5);
            this.dyMax_ = (int)(0.5 * (double)uh + 0.5);
        } else {
            this.dxMax_ = (int)(Math.min(0.5 * (double)uw, this.xmRS_) + 0.5);
            this.dyMax_ = (int)(Math.min(0.5 * (double)uh, this.ymRS_) + 0.5);
        }
        this.needsInverseRefresh_ = true;
    }

    protected void finishScaling() {
    }

    protected boolean isWithinMargins(Point2D.Double pt) {
        Objects.requireNonNull(pt, "Point cannot be null.");
        return this.isWithinMargins(pt.x, pt.y);
    }

    protected boolean isWithinMargins(double x, double y) {
        if (Double.isNaN(x) || Double.isNaN(y)) {
            return false;
        }
        return !(x < (double)this.lMargin_ - 0.6 || y < (double)this.tMargin_ - 0.6 || x > (double)this.rMargin_ + 0.6) && !(y > (double)this.bMargin_ + 0.6);
    }

    public Point2D.Double transformLL2XY(double lon, double lat) {
        Point2D.Double result = this.transformLL2XYIgnoreMargins(lon, lat);
        if (result == null || this.isWithinMargins(result)) {
            return result;
        }
        return null;
    }

    public Point2D.Double transformLL2XY(PointLL ll) {
        Point2D.Double result = this.transformLL2XYIgnoreMargins(ll);
        if (result == null || this.isWithinMargins(result)) {
            return result;
        }
        return null;
    }

    public Point transformLL2XYInt(double lon, double lat) {
        Point2D.Double p2d = this.transformLL2XY(lon, lat);
        if (p2d == null) {
            return null;
        }
        return new Point((int)p2d.x, (int)p2d.y);
    }

    protected abstract Point2D.Double transformLL2XYIgnoreMargins(double var1, double var3);

    public final Point2D.Double transformLL2XYIgnoreMargins(PointLL ll) {
        return this.transformLL2XY(ll.getLon(), ll.getLat());
    }

    public abstract PointLL transformXY2LL(double var1, double var3);

    public PointLL transformXY2LL(int x, int y) {
        int offset;
        if (!this.isWithinMargins(x, y)) {
            return null;
        }
        if (this.needsInverseRefresh_) {
            this.clearInverseArray();
        }
        if ((offset = y * this.outWidth_ + x) < 0 || offset >= this.invArray_.length || Double.isNaN(this.invArrayLon_[offset]) || Double.isNaN(this.invArrayLat_[offset])) {
            return null;
        }
        return new PointLL(this.invArrayLon_[offset], this.invArrayLat_[offset]);
    }

    public PointLL transformXY2LL(Point xy) {
        return this.transformXY2LL(xy.x, xy.y);
    }

    private double transformLLAngle2XYAngle(double lon, double lat, double azimuthAngleRad) {
        if (Math.abs(lat) >= 90.0) {
            return Double.NaN;
        }
        Point2D.Double dot1 = this.transformLL2XY(lon, lat);
        if (dot1 == null) {
            return Double.NaN;
        }
        Point2D.Double dot2 = this.transformLL2XY(lon, lat + 0.001);
        if (dot2 == null) {
            return Double.NaN;
        }
        Point2D.Double dot3 = this.transformLL2XY(lon + 0.001 * Math.sin(azimuthAngleRad) / Math.cos(Math.toRadians(lat)), lat + 0.001 * Math.cos(azimuthAngleRad));
        double dx3 = dot3.x - dot1.x;
        double dy3 = dot3.y - dot1.y;
        return Math.atan2(dx3, -dy3);
    }

    public double transformLLAngle2XYAngle(PointLL ll, double azimuthAngleRad) {
        return this.transformLLAngle2XYAngle(ll.getLon(), ll.getLat(), azimuthAngleRad);
    }

    protected double lonToLambda(double lon) {
        return MapUtils.normalizeMP180(lon - this.lambdaC_);
    }

    protected double lonToLambdaRad(double lon) {
        return Math.toRadians(this.lonToLambda(lon));
    }

    public final PointLL getCenter() {
        return new PointLL(this.lambdaC_, this.phiC_);
    }

    public void setCenter(double lon) {
        this.setCenter(lon, 0.0);
    }

    public void setCenter(PointLL pt) {
        this.setCenter(pt.getLon(), pt.getLat());
    }

    public void setCenter(double lon, double lat) {
        if (Math.abs(lat) > 90.0) {
            throw new IllegalArgumentException("Invalid latitude value.");
        }
        this.lambdaC_ = MapUtils.normalizeMP180(lon);
        this.lambdaCRad_ = Math.toRadians(this.lambdaC_);
        this.phiC_ = lat;
        this.phiCRad_ = Math.toRadians(this.phiC_);
        this.needsInverseRefresh_ = true;
    }

    public final boolean isZoomEnabled() {
        return this.zoomEnabled_;
    }

    protected final void setZoomEnabled(boolean b) {
        this.zoomEnabled_ = b;
        if (!this.zoomEnabled_) {
            this.zoom_ = 1.0;
        }
        this.autoscale();
    }

    public final double getZoom() {
        return this.zoomEnabled_ ? this.zoom_ : 1.0;
    }

    public final void setZoom(double zoom) {
        if (!this.zoomEnabled_) {
            LOGGER.debug("Projection does not accept a zoom level.");
            return;
        }
        if (zoom <= 0.0) {
            throw new IllegalArgumentException("Zoom must be greater than 0.0.");
        }
        this.zoom_ = zoom;
        this.autoscale();
    }

    public final Color getBackground() {
        return this.background_;
    }

    public final void setBackground(Color c) {
        this.background_ = c;
    }

    @Deprecated
    public final Color getForeground() {
        return this.getBorderColor();
    }

    public final Color getBorderColor() {
        return this.borderColor_;
    }

    @Deprecated
    public final void setForeground(Color c) {
        this.setBorderColor(c);
    }

    public final void setBorderColor(Color c) {
        this.borderColor_ = c;
    }

    public final BasicStroke getBorderStroke() {
        return this.borderStroke_;
    }

    public final void setBorderStroke(BasicStroke s) {
        this.borderStroke_ = s;
    }

    public final Color getGridColor() {
        return this.gridColor_;
    }

    public final void setGridColor(Color c) {
        this.gridColor_ = c;
    }

    public final BasicStroke getGridStroke() {
        return this.gridStroke_;
    }

    public final void setGridStroke(BasicStroke s) {
        this.gridStroke_ = s;
    }

    public final void setGridSpacing(double spacing) {
        this.setGridSpacing(spacing, spacing);
    }

    public final void setGridSpacing(double lonSpacing, double latSpacing) {
        this.setLonGridSpacing(lonSpacing);
        this.setLatGridSpacing(latSpacing);
    }

    public final double getLonGridSpacing() {
        return this.lonGridSpacing_;
    }

    public final void setLonGridSpacing(double spacing) {
        this.lonGridSpacing_ = Math.max(spacing, 0.0);
    }

    public final double getLatGridSpacing() {
        return this.latGridSpacing_;
    }

    public final void setLatGridSpacing(double spacing) {
        this.latGridSpacing_ = Math.max(spacing, 0.0);
    }

    public final boolean getLatGridOffset() {
        return this.latGridOffset_;
    }

    public final void setLatGridOffset(boolean offset) {
        this.latGridOffset_ = offset;
    }

    public final Font getGridFont() {
        return this.gridFont_;
    }

    public final void setGridFont(Font f) {
        this.gridFont_ = f;
    }

    public boolean canLabelGrid() {
        return false;
    }

    public boolean isGridLabeled() {
        return this.gridLabeled_ && this.canLabelGrid();
    }

    public void setGridLabeled(boolean b) {
        this.gridLabeled_ = b;
    }

    public int getGridLabelStep() {
        return this.gridLabelStep_;
    }

    public void setGridLabelStep(int step) {
        if (step < 0 || step > 5) {
            throw new IllegalArgumentException("Grid label step must be in range [0,5].");
        }
        this.gridLabelStep_ = step;
    }

    public final Color getPathColor() {
        return this.pathColor_;
    }

    public final void setPathColor(Color c) {
        this.pathColor_ = c;
    }

    public final BasicStroke getPathStroke() {
        return this.pathStroke_;
    }

    public final void setPathStroke(BasicStroke s) {
        this.pathStroke_ = s;
    }

    public final void setPointSymbol(SymbolID symbol, float size) {
        this.ptSymbol_ = symbol;
        this.ptSymbolSize_ = size;
    }

    public final void setSourceImage(BufferedImage source) {
        this.setSourceImage(source, LonLatEdges.ENTIRE_GLOBE);
    }

    public final void setSourceImage(BufferedImage source, LonLatEdges xbounds) {
        int hpxl;
        int wpxl;
        double right;
        double bottom;
        LonLatEdges bounds = source == null ? LonLatEdges.ENTIRE_GLOBE : xbounds;
        double top = bounds.getNorth();
        if (top == (bottom = bounds.getSouth())) {
            throw new IllegalArgumentException("Top and bottom latitude bounds cannot be equal.");
        }
        if (top > 90.0) {
            LOGGER.warn("Input map top cannot be > 90; resetting value to 90.");
            top = 90.0;
        }
        if (bottom < -90.0) {
            LOGGER.warn("Input map bottom cannot be < -90; resetting value to -90.");
            bottom = -90.0;
        }
        double left = bounds.getWest();
        left = MapUtils.normalizeMP180(left);
        for (right = bounds.getEast(); right <= left; right += 360.0) {
        }
        double wdeg = right - left;
        double hdeg = top - bottom;
        if (wdeg != this.srcWidthDeg_) {
            this.srcWidthDeg_ = wdeg;
            this.needsInverseRefresh_ = true;
        }
        if (hdeg != this.srcHeightDeg_) {
            this.srcHeightDeg_ = hdeg;
            this.needsInverseRefresh_ = true;
        }
        double clon = 0.5 * (right + left);
        double clat = 0.5 * (top + bottom);
        if (clon != this.srcCenterLon_) {
            this.srcCenterLon_ = clon;
            this.needsInverseRefresh_ = true;
        }
        if (clat != this.srcCenterLat_) {
            this.srcCenterLat_ = clat;
            this.needsInverseRefresh_ = true;
        }
        if (source == null) {
            this.srcPixels_ = null;
            wpxl = 2;
            hpxl = 2;
        } else {
            wpxl = source.getWidth(null);
            hpxl = source.getHeight(null);
            this.srcPixels_ = ImageUtils.getARGBPixels(source);
        }
        if (wpxl != this.srcWidth_) {
            this.srcWidth_ = wpxl;
            this.needsInverseRefresh_ = true;
        }
        if (hpxl != this.srcHeight_) {
            this.srcHeight_ = hpxl;
            this.needsInverseRefresh_ = true;
        }
        this.srcCenterX_ = 0.5 * (double)wpxl;
        this.srcCenterY_ = 0.5 * (double)hpxl;
        this.srcPixelPerDegLon_ = (double)wpxl / wdeg;
        this.srcPixelPerDegLat_ = (double)hpxl / hdeg;
    }

    public final int getExtraParamCount() {
        if (this.extraParams_ == null) {
            return 0;
        }
        return this.extraParams_.size();
    }

    public final ProjExtraParameter getParameter(int pid) {
        if (this.extraParams_ == null) {
            throw new IllegalArgumentException(this.getName() + " projection has no additional params");
        }
        if (pid < 0 || pid >= this.extraParams_.size()) {
            throw new IllegalArgumentException("Param ID " + pid + " out of range for " + this.getName());
        }
        return this.extraParams_.get(pid);
    }

    @Override
    public void parameterChanged(ProjParameterEvent e) {
    }

    protected final void addParameter(ProjExtraParameter param) {
        if (this.extraParams_ == null) {
            this.extraParams_ = new ArrayList(5);
        }
        this.extraParams_.add(param);
        param.addParameterListener(this);
        try {
            this.parameterChanged(new ProjParameterEvent(param));
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    protected void updateParamComponents() {
    }

    public final ProjParamComponent getParamComponent(int pid) {
        ProjExtraParameter xp = this.extraParams_.get(pid);
        this.updateParamComponents();
        if (xp == null) {
            return null;
        }
        return this.getParamComponent(xp);
    }

    public final ProjParamComponent getParamComponent(ProjExtraParameter xp) {
        if (xp == null) {
            return null;
        }
        if (this.extraParams_.indexOf(xp) < 0) {
            return null;
        }
        ProjParamComponent comp = paramGuiHash_.get(xp);
        if (comp != null) {
            return comp;
        }
        if (xp instanceof ProjDoubleParameter) {
            comp = new ProjDoubleComponent((ProjDoubleParameter)xp);
        } else if (xp instanceof ProjBooleanParameter) {
            comp = new ProjBooleanComponent((ProjBooleanParameter)xp);
        } else if (xp instanceof ProjIntegerParameter) {
            comp = new ProjIntegerComponent((ProjIntegerParameter)xp);
        } else if (xp instanceof ProjListParameter) {
            comp = new ProjListComponent((ProjListParameter)xp);
        }
        if (comp != null) {
            paramGuiHash_.put(xp, comp);
            return comp;
        }
        throw new RuntimeException("No GUI component defined for " + xp.getClass());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void paintMap(Graphics2D g2d) {
        AbstractProjection abstractProjection = this;
        synchronized (abstractProjection) {
            block13: {
                if (this.background_ == null) {
                    Composite oldComp = g2d.getComposite();
                    g2d.setComposite(AlphaComposite.Clear);
                    g2d.fillRect(0, 0, this.outWidth_, this.outHeight_);
                    g2d.setComposite(oldComp);
                } else {
                    g2d.setColor(this.background_);
                    g2d.fillRect(0, 0, this.outWidth_, this.outHeight_);
                }
                if (this.srcPixels_ == null) {
                    return;
                }
                int backgroundRGB = this.background_ == null ? 0 : this.background_.getRGB();
                int[] dstPixels = new int[this.outWidth_ * this.outHeight_];
                if (this.needsInverseRefresh_) {
                    this.clearInverseArray();
                }
                int offset = 0;
                int rowOffset = 0;
                for (int row = 0; row < this.outHeight_; ++row) {
                    rowOffset = row * this.outWidth_;
                    for (int col = 0; col < this.outWidth_; ++col) {
                        offset = rowOffset + col;
                        if (this.invArray_[offset] >= 0 && this.invArray_[offset] < this.srcPixels_.length) {
                            if (this.srcPixels_[this.invArray_[offset]] == 0) {
                                dstPixels[offset] = backgroundRGB;
                                continue;
                            }
                            dstPixels[offset] = this.srcPixels_[this.invArray_[offset]];
                            continue;
                        }
                        dstPixels[offset] = backgroundRGB;
                    }
                }
                try {
                    g2d.drawImage(Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(this.outWidth_, this.outHeight_, dstPixels, 0, this.outWidth_)), 0, 0, null);
                }
                catch (Exception exc) {
                    LOGGER.error("MemoryImageSource error");
                    if (!LOGGER.isTraceEnabled()) break block13;
                    exc.printStackTrace();
                }
            }
        }
    }

    public final void drawMapVectorized(Graphics2D g2d) {
        if (this.srcPixels_ == null && this.background_ != null) {
            g2d.setColor(this.background_);
            g2d.fillRect(0, 0, this.outWidth_, this.outHeight_);
            return;
        }
        int backgroundRGB = this.background_ == null ? 0 : this.background_.getRGB();
        int[] dstPixels = new int[this.outWidth_ * this.outHeight_];
        if (this.needsInverseRefresh_) {
            this.clearInverseArray();
        }
        int offset = 0;
        int rowOffset = 0;
        for (int row = 0; row < this.outHeight_; ++row) {
            rowOffset = row * this.outWidth_;
            for (int col = 0; col < this.outWidth_; ++col) {
                offset = rowOffset + col;
                dstPixels[offset] = this.invArray_[offset] >= 0 && this.invArray_[offset] < this.srcPixels_.length ? this.srcPixels_[this.invArray_[offset]] : backgroundRGB;
            }
        }
        g2d.setStroke(GraphicUtils.squareMiterStroke(5.0f));
        ArrayList<ColorArea> areas = new ArrayList<ColorArea>(3600);
        for (int row = 0; row < this.outHeight_; ++row) {
            rowOffset = row * this.outWidth_;
            int first = 0;
            for (int col = 1; col < this.outWidth_ - 1; ++col) {
                offset = rowOffset + col;
                if (dstPixels[offset] == dstPixels[rowOffset + first]) continue;
                Rectangle2D.Double area = new Rectangle2D.Double(first, row, col - first, 1.0);
                areas.add(new ColorArea(dstPixels[rowOffset + first], new Area(area)));
                first = col;
            }
            Rectangle2D.Double area = new Rectangle2D.Double(first, row, (double)this.outWidth_ - 1.0 - (double)first, 1.0);
            areas.add(new ColorArea(dstPixels[rowOffset + first], new Area(area)));
            if (areas.size() <= 3000) continue;
            this.drawAreaVectorized(g2d, areas);
            areas.clear();
        }
        this.drawAreaVectorized(g2d, areas);
    }

    private void drawAreaVectorized(Graphics2D g2d, ArrayList<ColorArea> areas) {
        while (!areas.isEmpty()) {
            ColorArea carea = areas.remove(areas.size() - 1);
            int colorval = carea.colorval_;
            Area area = carea.area_;
            for (int i = areas.size() - 1; i >= 0; --i) {
                carea = areas.get(i);
                if (carea.colorval_ != colorval) continue;
                area.add(carea.area_);
                areas.remove(i);
            }
            Color cc = new Color(colorval);
            if (cc.equals(this.background_)) continue;
            g2d.setColor(cc);
            g2d.fill(area);
            g2d.draw(area);
        }
    }

    public final void drawGrid(Graphics2D g2d) {
        if (this.gridStroke_ == null || this.gridStroke_.getLineWidth() == 0.0f) {
            return;
        }
        if (this.gridColor_ == null) {
            return;
        }
        g2d.setStroke(this.gridStroke_);
        g2d.setColor(this.gridColor_);
        g2d.setFont(this.gridFont_);
        Shape oldclip = g2d.getClip();
        if (oldclip == null) {
            double uw = (double)(this.rMargin_ - this.lMargin_) + 0.5;
            double uh = (double)(this.bMargin_ - this.tMargin_) + 0.5;
            Rectangle2D.Double newclip = new Rectangle2D.Double((double)this.lMargin_ - 0.25, (double)this.tMargin_ - 0.25, uw, uh);
            g2d.setClip(newclip);
        }
        this.drawParallels(g2d);
        this.drawMeridians(g2d);
        g2d.setClip(oldclip);
    }

    private void drawParallels(Graphics2D g2d) {
        if (this.latGridSpacing_ <= 0.0) {
            return;
        }
        int steps = (int)(90.0 / this.latGridSpacing_);
        double halfSpacing = 0.5 * this.latGridSpacing_;
        if ((double)steps * this.latGridSpacing_ < 90.0) {
            ++steps;
        }
        for (int j = 0; j < steps; ++j) {
            double lat = this.latGridSpacing_ * (double)j;
            if (this.latGridOffset_) {
                lat += halfSpacing;
            }
            if (lat >= 90.0) break;
            boolean showLabel = this.canLabelGrid() && this.gridLabeled_ && this.gridLabelStep_ > 0 && j % this.gridLabelStep_ == 0;
            String label = showLabel ? AbstractProjection.trimDecimalZero(GRID_LABEL_FORMATTER.sprintfx(lat)) + "N" : null;
            this.drawParallel(g2d, lat, label);
            if (!(lat > 0.0)) continue;
            label = showLabel ? AbstractProjection.trimDecimalZero(GRID_LABEL_FORMATTER.sprintfx(lat)) + "S" : null;
            this.drawParallel(g2d, -lat, label);
        }
    }

    protected void drawParallel(Graphics2D g2d, double lat, String label) {
        double llon = this.lambdaC_ - 179.99999;
        double istep = 0.5;
        int isteps = 720;
        PointLL[] latPoints = new PointLL[721];
        for (int ilon = 0; ilon <= 720; ++ilon) {
            double lon = llon + 0.5 * (double)ilon;
            if (latPoints[ilon] == null) {
                latPoints[ilon] = new PointLL(lon, lat);
                continue;
            }
            latPoints[ilon].setLat(lat);
        }
        ProjectionUtils.drawBezier(g2d, this.translatePath(latPoints));
    }

    private void drawMeridians(Graphics2D g2d) {
        if (this.lonGridSpacing_ <= 0.0) {
            return;
        }
        int steps = (int)(180.0 / this.lonGridSpacing_);
        for (int i = 0; i <= steps; ++i) {
            double lon = this.lonGridSpacing_ * (double)i;
            boolean showLabel = this.canLabelGrid() && this.gridLabeled_ && this.gridLabelStep_ > 0 && i % this.gridLabelStep_ == 0;
            String label = showLabel ? AbstractProjection.trimDecimalZero(GRID_LABEL_FORMATTER.sprintfx(lon)) + "E" : null;
            this.drawMeridian(g2d, lon, label);
            if (!(lon > 0.0) || !(lon < 180.0)) continue;
            label = showLabel ? AbstractProjection.trimDecimalZero(GRID_LABEL_FORMATTER.sprintfx(lon)) + "W" : null;
            this.drawMeridian(g2d, -lon, label);
        }
    }

    protected void drawMeridian(Graphics2D g2d, double lon, String label) {
        double jstep = 0.125;
        int jsteps = 1440;
        PointLL[] llpts = new PointLL[1441];
        for (int jj = 0; jj <= 1440; ++jj) {
            double lat = 90.0 - 0.125 * (double)jj;
            if (jj == 0) {
                lat = 89.9999999999;
            } else if (jj == 1440) {
                lat = -89.9999999999;
            }
            llpts[jj] = new PointLL(lon, lat);
        }
        Point2D.Double[] ppp = this.translatePath(llpts);
        ProjectionUtils.drawBezier(g2d, ppp);
    }

    public final void drawPath(Graphics2D g2d, PointLL ... llPoints) {
        this.drawPath(g2d, llPoints, (String)null, (Font)null);
    }

    public final void drawPath(Graphics2D g2d, OutlineOverlay path) {
        double[] bb1;
        double[] bb0 = this.getBounds().asArray();
        if (!AbstractProjection.rectBoundsIntersect(bb0, bb1 = path.getBounds().asArray())) {
            return;
        }
        int isize = path.size();
        for (int i = 0; i < isize; ++i) {
            OutlineArea area = path.getArea(i);
            bb1 = area.getBounds().asArray();
            if (!AbstractProjection.rectBoundsIntersect(bb0, bb1)) continue;
            int jsize = area.size();
            for (int j = 0; j < jsize; ++j) {
                OutlineSegment segment = area.getSegment(j);
                PointLL[] points = segment.getPoints();
                if (segment.isShapes()) {
                    this.drawPath(g2d, points, (String)null, (Font)null);
                    continue;
                }
                if (!segment.isPoints()) continue;
                this.markPoints(g2d, points, this.pathStroke_, this.pathColor_, null);
            }
        }
    }

    public final void drawPath(Graphics2D g2d, PointLL[] llPoints, String label, Font lfont) {
        if (llPoints == null || llPoints.length == 0) {
            return;
        }
        if (this.pathStroke_ == null || this.pathStroke_.getLineWidth() == 0.0f) {
            return;
        }
        if (this.pathColor_ == null || this.pathColor_.getAlpha() == 0) {
            return;
        }
        g2d.setStroke(this.pathStroke_);
        g2d.setColor(this.pathColor_);
        int isize = llPoints.length;
        int maxPts = 50000;
        if (isize > 66500) {
            int numSects = isize / 50000 + 1;
            for (int i = 0; i < numSects; ++i) {
                int istart = i * 50000;
                int ilast = Math.min((i + 1) * 50000, isize - 1);
                if (istart < isize) {
                    PointLL[] range = Arrays.copyOfRange(llPoints, istart, ilast);
                    if (i == 0) {
                        this.drawPathSection(g2d, llPoints, label, lfont);
                        continue;
                    }
                    this.drawPathSection(g2d, llPoints, null, null);
                    continue;
                }
                break;
            }
        } else {
            this.drawPathSection(g2d, llPoints, label, lfont);
        }
    }

    protected void drawPathSection(Graphics2D g2d, PointLL[] llPoints, String label, Font lfont) {
        g2d.setStroke(this.pathStroke_);
        g2d.setColor(this.pathColor_);
        Point2D.Double[] xyPoints = this.translatePath(llPoints);
        Polygon lpoly = null;
        if (label != null && lfont != null && label.length() > 0 && xyPoints.length > 50) {
            g2d.setFont(lfont);
            FontMetrics fm = g2d.getFontMetrics();
            int twidth = fm.stringWidth(label);
            for (int i = xyPoints.length / 4; i < xyPoints.length - 10; i += 5) {
                boolean goodpt = true;
                for (int j = -5; j < 6; ++j) {
                    if (xyPoints[i + j] != null) continue;
                    goodpt = false;
                    break;
                }
                if (!goodpt) continue;
                Point2D.Double p0 = xyPoints[i];
                Point2D.Double pa = xyPoints[i - 3];
                Point2D.Double pb = xyPoints[i + 3];
                double dy = pb.y - pa.y;
                double dx = pb.x - pa.x;
                double rad = Math.atan2(dy, dx);
                if (rad > 1.5707963267948966) {
                    rad -= Math.PI;
                }
                if (rad < -1.5707963267948966) {
                    rad += Math.PI;
                }
                float px = (float)p0.x;
                float py = (float)p0.y;
                float hw = (float)(0.5 * (double)twidth);
                float hh = (float)(0.5 * (double)fm.getAscent());
                Point[] pc = this.findLabelCorners(px, py, twidth, fm.getAscent(), (float)rad);
                if (this.transformXY2LL(pc[0]) == null || this.transformXY2LL(pc[1]) == null || this.transformXY2LL(pc[2]) == null || this.transformXY2LL(pc[3]) == null) continue;
                g2d.translate(px, py);
                g2d.rotate(rad);
                g2d.drawString(label, -hw, hh);
                g2d.rotate(-rad);
                g2d.translate(-px, -py);
                lpoly = new Polygon(new int[]{pc[0].x, pc[1].x, pc[2].x, pc[3].x}, new int[]{pc[0].y, pc[1].y, pc[2].y, pc[3].y}, 4);
                break;
            }
        }
        Path2D.Double path = new Path2D.Double();
        for (Point2D.Double pt : xyPoints) {
            if (pt == null) {
                if (path.getCurrentPoint() != null) {
                    g2d.draw(path);
                }
                path.reset();
                continue;
            }
            float px = (float)pt.x;
            float py = (float)pt.y;
            if (lpoly != null && lpoly.contains(px, py)) {
                if (path.getCurrentPoint() != null) {
                    g2d.draw(path);
                }
                path.reset();
                continue;
            }
            if (path.getCurrentPoint() == null) {
                path.moveTo(px, py);
                continue;
            }
            path.lineTo(px, py);
        }
        if (path.getCurrentPoint() != null) {
            g2d.draw(path);
        }
    }

    public void markPoints(Graphics2D g2d, PointLL[] llPoints, BasicStroke stroke, Color edgeColor, Color fillColor) {
        Point2D.Double[] xyPoints;
        if (llPoints == null || llPoints.length == 0) {
            return;
        }
        if (edgeColor == null) {
            return;
        }
        if (stroke == null || stroke.getLineWidth() == 0.0f) {
            return;
        }
        Path2D.Float symbol = SymbolFactory.getSymbol(this.ptSymbol_, this.ptSymbolSize_);
        if (symbol == null) {
            return;
        }
        Rectangle2D sbounds = symbol.getBounds2D();
        double swidth = sbounds.getWidth();
        double sheight = sbounds.getHeight();
        if (swidth == 0.0 || sheight == 0.0) {
            return;
        }
        double cx = sbounds.getCenterX();
        double cy = sbounds.getCenterY();
        boolean fill = fillColor != null && this.ptSymbol_.isClosed();
        g2d.setStroke(stroke);
        for (Point2D.Double pt : xyPoints = this.translatePath(llPoints)) {
            if (pt == null) continue;
            g2d.translate(pt.x - cx, pt.y - cy);
            if (fill) {
                g2d.setColor(fillColor);
                g2d.fill(symbol);
            }
            if (edgeColor != null) {
                g2d.setColor(edgeColor);
                g2d.draw(symbol);
            }
            g2d.translate(-(pt.x - cx), -(pt.y - cy));
        }
    }

    public void markPoint(Graphics2D g2d, PointLL llPoint, BasicStroke stroke, Color edgeColor, Color fillColor) {
        if (llPoint == null) {
            return;
        }
        if (edgeColor == null && fillColor == null) {
            return;
        }
        Path2D.Float symbol = SymbolFactory.getSymbol(this.ptSymbol_, this.ptSymbolSize_);
        if (symbol == null) {
            return;
        }
        Rectangle2D sbounds = symbol.getBounds2D();
        double swidth = sbounds.getWidth();
        double sheight = sbounds.getHeight();
        if (swidth == 0.0 || sheight == 0.0) {
            return;
        }
        Point2D.Double xy = this.transformLL2XY(llPoint.getLon(), llPoint.getLat());
        if (xy == null) {
            return;
        }
        double cx = sbounds.getCenterX();
        double cy = sbounds.getCenterY();
        g2d.translate(xy.x - cx, xy.y - cy);
        if (fillColor != null && this.ptSymbol_.isClosed()) {
            g2d.setColor(fillColor);
            g2d.fill(symbol);
        }
        if (edgeColor != null && stroke != null && stroke.getLineWidth() != 0.0f) {
            g2d.setStroke(stroke);
            g2d.setColor(edgeColor);
            g2d.draw(symbol);
        }
        g2d.translate(-(xy.x - cx), -(xy.y - cy));
    }

    private final Point[] findLabelCorners(float px, float py, float w, float h, float angle) {
        float hw = (float)(0.6 * (double)w);
        float hh = (float)(0.6 * (double)h);
        float cosA = (float)Math.cos(angle);
        float sinA = (float)Math.sin(angle);
        int px1 = (int)(px + hw * cosA - hh * sinA);
        int px2 = (int)(px + hw * cosA + hh * sinA);
        int px3 = (int)(px - hw * cosA + hh * sinA);
        int px4 = (int)(px - hw * cosA - hh * sinA);
        int py1 = (int)(py + hw * sinA + hh * cosA);
        int py2 = (int)(py + hw * sinA - hh * cosA);
        int py3 = (int)(py - hw * sinA - hh * cosA);
        int py4 = (int)(py - hw * sinA + hh * cosA);
        return new Point[]{new Point(px1, py1), new Point(px2, py2), new Point(px3, py3), new Point(px4, py4)};
    }

    public void drawBorder(Graphics2D g2d) {
        BasicStroke stroke = this.getBorderStroke();
        if (stroke == null || stroke.getLineWidth() == 0.0f) {
            return;
        }
        Color color = this.getForeground();
        if (color == null) {
            return;
        }
        g2d.setStroke(stroke);
        g2d.setColor(color);
        this.drawBorderLines(g2d);
    }

    protected abstract void drawBorderLines(Graphics2D var1);

    private void clearInverseArray() {
        if (this.invArray_ == null) {
            this.invArray_ = new int[this.outWidth_ * this.outHeight_];
            this.invArrayLon_ = new double[this.outWidth_ * this.outHeight_];
            this.invArrayLat_ = new double[this.outWidth_ * this.outHeight_];
        }
        for (int j = 0; j < this.outHeight_; ++j) {
            for (int i = 0; i < this.outWidth_; ++i) {
                int offset = j * this.outWidth_ + i;
                this.invArray_[offset] = Integer.MIN_VALUE;
                this.invArrayLon_[offset] = Double.NaN;
                this.invArrayLat_[offset] = Double.NaN;
            }
        }
        this.calculateInverseArray();
        this.needsInverseRefresh_ = false;
    }

    protected abstract void calculateInverseArray();

    protected void setInverseArrayLocation(int col, int row, double lon, double lat) {
        if (row < 0 || row >= this.outHeight_ || col < 0 || col >= this.outWidth_) {
            return;
        }
        int index = row * this.outWidth_ + col;
        this.invArrayLon_[index] = MapUtils.normalize360(lon);
        this.invArrayLat_[index] = lat;
        int srcX = this.getSrcPixelX(lon);
        if (srcX < 0 || srcX >= this.srcWidth_) {
            return;
        }
        int srcY = this.getSrcPixelY(lat);
        if (srcY < 0 || srcY >= this.srcHeight_) {
            return;
        }
        int srcIndex = srcY * this.srcWidth_ + srcX;
        if (srcIndex > -1) {
            this.invArray_[index] = srcIndex;
        }
    }

    private final int getSrcPixelX(double lon) {
        if (this.srcWidth_ < 1) {
            return -1;
        }
        double dlon = MapUtils.normalizeMP180(lon - this.srcCenterLon_);
        int x = (int)(this.srcCenterX_ + this.srcPixelPerDegLon_ * dlon);
        if (x < 0 || x >= this.srcWidth_) {
            return -1;
        }
        return x;
    }

    private final int getSrcPixelY(double lat) {
        if (this.srcHeight_ < 1) {
            return -1;
        }
        int y = (int)(this.srcCenterY_ - this.srcPixelPerDegLat_ * (lat - this.srcCenterLat_));
        if (y < 0 || y >= this.srcHeight_) {
            return -1;
        }
        return y;
    }

    protected final Point2D.Double[] translatePath(PointLL ... llPoints) {
        Point2D.Double prevDot = null;
        PointLL prevLL = null;
        Point2D.Double midDot = null;
        ArrayList<Point2D.Double> v = new ArrayList<Point2D.Double>(llPoints.length + 50);
        for (PointLL ll : llPoints) {
            Point2D.Double edgeDot;
            Object[] edge;
            if (ll == null) {
                if (v.size() > 0 && v.get(v.size() - 1) != null) {
                    v.add(null);
                }
                prevDot = null;
                prevLL = null;
                continue;
            }
            double lon = ll.getLon();
            double lat = ll.getLat();
            Point2D.Double dot = this.transformLL2XY(lon, lat);
            if (prevLL == null) {
                if (dot == null) {
                    prevDot = null;
                    prevLL = ll.copy();
                    continue;
                }
                v.add(dot);
                prevDot = dot;
                prevLL = ll.copy();
                continue;
            }
            if (dot == null && prevDot == null) {
                if (v.size() > 0 && v.get(v.size() - 1) != null) {
                    v.add(null);
                }
                prevDot = null;
                prevLL = ll.copy();
                continue;
            }
            if (dot != null && prevDot != null) {
                Object[] mid = this.findMidpoint(ll, prevLL);
                midDot = mid == null ? null : (Point2D.Double)mid[1];
                if (midDot == null) {
                    v.add(null);
                } else {
                    double dx1 = midDot.x - prevDot.x;
                    double dx2 = dot.x - midDot.x;
                    double dx12 = dot.x - prevDot.x;
                    double dy1 = midDot.y - prevDot.y;
                    double dy2 = dot.y - midDot.y;
                    double dy12 = dot.y - prevDot.y;
                    double d1 = Math.hypot(dx1, dy1);
                    double d2 = Math.hypot(dx2, dy2);
                    double d12 = Math.hypot(dx12, dy12);
                    if (d12 > 1.0 && d1 / d12 < 0.25) {
                        v.add(midDot);
                        v.add(null);
                    } else if (d12 > 1.0 && d2 / d12 < 0.25) {
                        v.add(null);
                        v.add(midDot);
                    } else if (dx1 * dx2 < 0.0 && (d1 > d12 || d2 > d12)) {
                        v.add(null);
                    }
                }
                v.add(dot);
                prevDot = dot;
                prevLL = ll.copy();
                continue;
            }
            if (dot == null && prevDot != null) {
                Point2D.Double edgeDot2;
                edge = this.findEdgePoint(0, prevLL, ll);
                if (edge != null && (edgeDot2 = (Point2D.Double)edge[1]) != null) {
                    v.add(edgeDot2);
                }
                v.add(null);
                prevDot = null;
                prevLL = ll.copy();
                continue;
            }
            edge = this.findEdgePoint(0, ll, prevLL);
            if (edge != null && (edgeDot = (Point2D.Double)edge[1]) != null) {
                v.add(edgeDot);
            }
            v.add(dot);
            prevDot = dot;
            prevLL = ll.copy();
        }
        return v.toArray(new Point2D.Double[0]);
    }

    private final Object[] findMidpoint(PointLL ll1, PointLL ll2) {
        double lon1 = ll1.getLon();
        double lon2 = ll2.getLon();
        PointLL ll = new PointLL();
        if (Math.abs(lon1 - lon2) < 180.0) {
            ll.setLon(0.5 * (lon1 + lon2));
        } else {
            ll.setLon(0.5 * (lon1 + lon2) + 180.0);
        }
        ll.setLat(0.5 * (ll1.getLat() + ll2.getLat()));
        return new Object[]{ll, this.transformLL2XY(ll)};
    }

    private final Object[] findEdgePoint(int iternum, PointLL insideLL, PointLL outsideLL) {
        if (iternum > 15) {
            return null;
        }
        Object[] o = this.findMidpoint(insideLL, outsideLL);
        PointLL ll = (PointLL)o[0];
        Point2D.Double dot = (Point2D.Double)o[1];
        if (dot == null) {
            return this.findEdgePoint(iternum + 1, insideLL, ll);
        }
        Point2D.Double insideDot = this.transformLL2XY(insideLL);
        double dx = Math.abs(dot.x - insideDot.x);
        double dy = Math.abs(dot.y - insideDot.y);
        if (dx < 2.0 && dy < 2.0) {
            return new Object[]{ll, dot};
        }
        if (dx > 50.0 || dy > 50.0) {
            return this.findEdgePoint(iternum + 1, insideLL, ll);
        }
        return this.findEdgePoint(iternum + 1, ll, outsideLL);
    }

    public LonLatEdges getBounds() {
        return LonLatEdges.ENTIRE_GLOBE;
    }

    public final String getName() {
        return this.name_;
    }

    public final String getNormalizedName() {
        return Normalizer.normalize(this.name_, Normalizer.Form.NFD).replaceAll("\\p{M}", "");
    }

    public final int getProperties() {
        return this.properties_;
    }

    public boolean isRecenterableLon() {
        return true;
    }

    public boolean isRecenterableLat() {
        return this.hasProperty(0x200000);
    }

    public final boolean hasProperty(int prop) {
        return (this.properties_ & prop) != 0;
    }

    private static final boolean rectBoundsIntersect(double[] bb0, double[] bb1) {
        Rectangle2D.Double rect0;
        double temp;
        if (bb0[1] > bb0[3]) {
            temp = bb0[1];
            bb0[1] = bb0[3];
            bb0[3] = temp;
        }
        if (bb1[1] > bb1[3]) {
            temp = bb1[1];
            bb1[1] = bb1[3];
            bb1[3] = temp;
        }
        if ((rect0 = new Rectangle2D.Double(bb0[0], bb0[1], bb0[2] - bb0[0], bb0[3] - bb0[1])).contains(bb1[0], bb1[1]) || rect0.contains(bb1[0], bb1[3]) || rect0.contains(bb1[2], bb1[1]) || rect0.contains(bb1[2], bb1[3])) {
            return true;
        }
        Rectangle2D.Double rect1 = new Rectangle2D.Double(bb1[0], bb1[1], bb1[2] - bb1[0], bb1[3] - bb1[1]);
        if (rect1.contains(bb0[0], bb0[1]) || rect1.contains(bb0[0], bb0[3]) || rect1.contains(bb0[2], bb0[1]) || rect1.contains(bb0[2], bb0[3])) {
            return true;
        }
        boolean intersects = rect1.intersects(rect0);
        if (intersects) {
            return true;
        }
        if ((rect0.outcode(bb1[0], bb1[1]) & 1) != 0) {
            rect0.x = bb0[0] - 360.0;
            rect0.y = bb0[1];
            rect0.width = bb0[2] - bb0[0];
            rect0.height = bb0[3] - bb0[1];
        } else if ((rect0.outcode(bb1[0], bb1[1]) & 4) != 0) {
            rect0.x = bb0[0] + 360.0;
            rect0.y = bb0[1];
            rect0.width = bb0[2] - bb0[0];
            rect0.height = bb0[3] - bb0[1];
        } else {
            return false;
        }
        if (rect0.contains(bb1[0], bb1[1]) || rect0.contains(bb1[0], bb1[3]) || rect0.contains(bb1[2], bb1[1]) || rect0.contains(bb1[2], bb1[3])) {
            return true;
        }
        if (rect1.contains(bb0[0], bb0[1]) || rect1.contains(bb0[0], bb0[3]) || rect1.contains(bb0[2], bb0[1]) || rect1.contains(bb0[2], bb0[3])) {
            return true;
        }
        intersects = rect1.intersects(rect0);
        return intersects;
    }

    private static final String trimDecimalZero(String label) {
        char decimalChar = GRID_LABEL_FORMATTER.getDecimalChar();
        if (label.endsWith(decimalChar + "0")) {
            return label.substring(0, label.length() - 2);
        }
        return label;
    }

    private class ColorArea {
        int colorval_;
        Area area_;

        ColorArea(int colorval, Area area) {
            this.colorval_ = colorval;
            this.area_ = area;
        }
    }
}

